#!/usr/bin/env python3
"""
make_kernel_from_eigs.py

Although the fluctuation simulation does not use a kernel in its pipeline,
this helper script is provided for completeness.  It reads eigenvalues from
`kernel_eigs.csv` in the data directory, extracts the `rho` column, and
constructs a 1‑D kernel of length `2*L^2` matching the configured lattice
size.  The resulting array is saved to `data/kernel.npy`.
"""

import os
import yaml
import numpy as np
import pandas as pd


def load_config(config_path: str) -> dict:
    with open(config_path, "r") as f:
        return yaml.safe_load(f)


def main() -> None:
    repo_root = os.path.abspath(os.path.join(os.path.dirname(__file__), ".."))
    config_path = os.path.join(repo_root, "config.yaml")
    cfg = load_config(config_path)
    # Determine lattice size from config
    L = cfg.get("parameters", {}).get("lattice_size")
    if L is None:
        raise RuntimeError("Unable to determine lattice size from configuration")
    # load eigenvalue CSV
    csv_path = os.path.join(repo_root, "data", "kernel_eigs.csv")
    if not os.path.exists(csv_path):
        raise FileNotFoundError(csv_path)
    df = pd.read_csv(csv_path)
    if "rho" not in df.columns:
        raise KeyError("rho column not found in kernel_eigs.csv")
    rho = df["rho"].to_numpy(dtype=float)
    # Build kernel of length 2*L*L by tiling/truncating rho values
    n_links = 2 * L * L
    # tile or truncate
    if len(rho) < n_links:
        repeats = int(np.ceil(n_links / len(rho)))
        kernel = np.tile(rho, repeats)[:n_links]
    else:
        kernel = rho[:n_links]
    out_path = os.path.join(repo_root, "data", "kernel.npy")
    np.save(out_path, kernel)
    print(
        f"Saved kernel to {out_path} with shape {kernel.shape} and dtype {kernel.dtype}"
    )


if __name__ == "__main__":
    main()